// $Id: CFileManager.cpp,v 1.9 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CFileManager.hpp"
#include "../Testing/CTrace.hpp"
using Exponent::Testing::CTrace;
using Exponent::Host::CFileManager;

//	===========================================================================

#ifdef WIN32
	#include "shlwapi.h"
	#pragma comment (lib, "shlwapi.lib")
#else
	#include <sys/types.h>
    #include <sys/stat.h>
    #include <dirent.h>
	#include <unistd.h>
#endif

//	===========================================================================
bool CFileManager::doesFileExist(const CSystemString &filename)
{
	FILE *infile;
	if ((infile = fopen(filename.getString(), "r")) != NULL)
	{
		fclose(infile);
		return true;
	}
	return false;
}

//	===========================================================================
bool CFileManager::fileExistsInDirectory(const CSystemString &filename, const CSystemString &path)
{
	CSystemString fullPath = path.getString();
	fullPath.appendPath(filename.getString());
	return CFileManager::doesFileExist(fullPath);
}

//	===========================================================================
void CFileManager::getFilesInDirectory(const CSystemString &path, TStringCountedPointerArray *array)
{
#ifdef WIN32

	WIN32_FIND_DATA fileData;
	HANDLE fhFile = FindFirstFile(path.getString(), &fileData);

	if (fhFile == INVALID_HANDLE_VALUE)
	{
		return;
	}
	else
	{
		// Valid path and files in the path
		unsigned long test;
		do
		{
			test = fileData.dwFileAttributes;
			if(test != FILE_ATTRIBUTE_DIRECTORY)
			{
				array->addElement(new CString(fileData.cFileName));
			}
		}while(FindNextFile(fhFile, &fileData));
		FindClose(fhFile);
	}
#else
	// Get a handle to the directory
	chdir(".");
	DIR *directory = opendir(path.getString());
	if (directory != NULL)
	{
		// The directory entry that we will be iterating over
		dirent *directoryEntry;

		// Iterate until we reach the end of the folder
		while ((directoryEntry = readdir(directory)) != NULL)
		{
			if (directoryEntry->d_type != DT_DIR)
			{
				array->addElement(new CString(directoryEntry->d_name));
			}
		}

		// Finally we need to close the directory that we opened
		closedir(directory);
	}
#endif
}

//	===========================================================================
void CFileManager::getFilesInDirectory(TCountedPointerArray<CFileInformation> &array, const CSystemString &path)
{
#ifdef WIN32
	CSystemString thePath = path;
	//thePath.appendPath("*.*");
	thePath.appendPath("*");

	WIN32_FIND_DATA fileData;
	HANDLE fhFile = FindFirstFile(thePath.getString(), &fileData);

	if (fhFile == INVALID_HANDLE_VALUE)
	{
		char buffer[256];
		FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, 0, GetLastError(), 0, buffer, 256, NULL);
		CTrace::trace(path, "Failed to find file in directory");
		CTrace::trace(buffer);
		return;
	}
	else
	{
		// Store the main paths
		CSystemString folderPath = path;
		CSystemString actualFolderPath = folderPath;//(folderPath.getFolderPath());

		// Append a trailing slash
		if (actualFolderPath.characterAt(actualFolderPath.getNumberOfCharacters() - 2) != '\\')
		{
			actualFolderPath.appendSystemSeperator();
		}

		// Find the files
		do
		{
			if ((!(strcmp (fileData.cFileName, ".") == 0)) && (!(strcmp(fileData.cFileName, "..") == 0)))
			{
				CFileInformation *info = new CFileInformation(fileData);
				info->setFoldername(actualFolderPath);
				array.addElement(info);
			}
		}while(FindNextFile(fhFile, &fileData));
		FindClose(fhFile);
	}
#else
	// Get a handle to the directory
	chdir(".");
	DIR *directory = opendir(path.getString());
	if (directory != NULL)
	{
		// The directory entry that we will be iterating over
		dirent *directoryEntry;
		CSystemString folderPath = path;
		CString myExtension;
		//char *extension          = NULL;
		bool checkExtension      = false;
		if (folderPath.hasExtension())
		{
			myExtension    = folderPath.getTheExtension();
			//extension      = folderPath.getTheExtension();
			checkExtension = true;
		}
		CString actualFolderPath = folderPath;//(folderPath.getFolderPath());

		// Iterate until we reach the end of the folder
		while ((directoryEntry = readdir(directory)) != NULL)
		{
			if ((!(strcmp (directoryEntry->d_name, ".") == 0)) && (!(strcmp(directoryEntry->d_name, "..") == 0)))
			{
				CFileInformation *info = new CFileInformation(directoryEntry);
				info->setFoldername(actualFolderPath);
				if (checkExtension)
				{
					CSystemString theFileName = info->getFilename();
					if (theFileName.hasExtension(myExtension.getString()))
					{
						array.addElement(info);
					}
				}
				else
				{
					array.addElement(info);
				}
			}
		}

		//FREE_ARRAY_POINTER(extension);

		// Finally we need to close the directory that we opened
		closedir(directory);
	}
#endif
}

//	===========================================================================
long CFileManager::getNumberOfFilesInDirectory(const CSystemString &path)
{
#ifdef WIN32
	WIN32_FIND_DATA fileData;
	HANDLE fhFile = FindFirstFile(path.getString(), &fileData);

	if (fhFile == INVALID_HANDLE_VALUE)
	{
		return 0;
	}
	else
	{
		// Valid path and files in the path
		unsigned long test;
		long numFound = 0;
		do
		{
			test = fileData.dwFileAttributes;
			if (!((fileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) > 0))
			//if(test != FILE_ATTRIBUTE_DIRECTORY)
			{
				numFound++;
			}
		}while(FindNextFile(fhFile, &fileData));
		FindClose(fhFile);
		return numFound;
	}
	return 0;
#else
	// Get a handle to the directory
	chdir(".");
	DIR *directory = opendir(path.getString());
	if (directory != NULL)
	{
		// The directory entry that we will be iterating over
		dirent *directoryEntry;
		long numFound = 0;

		// Iterate until we reach the end of the folder
		while ((directoryEntry = readdir(directory)) != NULL)
		{
			if (directoryEntry->d_type != DT_DIR)
			{
				numFound++;
			}
		}

		// Finally we need to close the directory that we opened
		closedir(directory);

		return numFound;
	}
	return 0;
#endif
}

//	===========================================================================
void CFileManager::getApplicationDirectory(CSystemString &string, bool addTrailingSlashes)
{
#ifdef WIN32
	char filename[_MAX_PATH];
	GetModuleFileName(NULL, filename, _MAX_PATH);
	PathRemoveFileSpec(filename);

	if (addTrailingSlashes)
	{
		PathAddBackslash(filename);
	}

	string = filename;
#else
	char buffer[_MAX_PATH];
	getcwd(buffer, _MAX_PATH * sizeof(char));
	string = buffer;
#endif
}

//	===========================================================================
bool CFileManager::createNewFolder(const CSystemString &string)
{
	CSystemString str = string;
	if (!str.hasExtension())
	{
#ifdef WIN32
		if (CreateDirectory(str.getString(), NULL))
		{
			return true;
		}
#else
		if (mkdir(string.getString(), S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IWGRP | S_IXGRP | S_IROTH | S_IWOTH | S_IXOTH) == 0)
		{
			return true;
		}
#endif
	}
	return false;
}

//	===========================================================================
bool CFileManager::isFolderPath(const CSystemString &path)
{
#ifdef WIN32
	return PathIsDirectory(path.getString()) != FALSE;
#else
	bool retValue = false;
	
	// Get the CFStringRef
	CFStringRef fileNameRef = path.getStringAsCFStringRef();
	
	// If its valid
	if (fileNameRef)
	{
		// Create the URL
		CFURLRef url = CFURLCreateWithFileSystemPath(kCFAllocatorDefault, fileNameRef, kCFURLPOSIXPathStyle, false);
		
		// Check the url
		if (url)
		{
			// Convrt to an FsRef
			FSRef fsRef;
			if (CFURLGetFSRef(url, &fsRef))
			{
				// Ge the catalog info
				FSCatalogInfo info;
				memset(&info, 0, sizeof(FSCatalogInfo));
				FSSpec fsSpec;
				if (FSGetCatalogInfo(&fsRef, kFSCatInfoGettableInfo, &info, NULL, &fsSpec, NULL) == noErr)
				{
					retValue = (bool)(kFSNodeIsDirectoryMask & info.nodeFlags);
				}
			}
			
			// Release
			CFRelease(url);
		}
		
		// Release
		CFRelease(fileNameRef);
	}
	
	// We are done!
	return retValue;
#endif
}

//	===========================================================================
void CFileManager::sortDirectoryContents(TCountedPointerArray<CFileInformation> &array)
{
	array.sortArray((TPointerCollection<CFileInformation>::qsortCompare)CFileInformation::compareFiles);
}